<?php 
$page_title = 'Explore All Tours';
require_once 'includes/header.php'; 

// --- PAGE-SPECIFIC QUERIES ---

// 1. Build the filter query
$where_clauses = ["status = 'published'", "category = 'Tour'"];
$params = [];
$types = '';

$search_query = trim($_GET['q'] ?? '');
if (!empty($search_query)) {
    $where_clauses[] = "(title LIKE ? OR location LIKE ?)";
    $like_query = "%{$search_query}%";
    array_push($params, $like_query, $like_query);
    $types .= 'ss';
}

$location_filter = trim($_GET['location'] ?? '');
if (!empty($location_filter)) {
    $where_clauses[] = "location = ?";
    $params[] = $location_filter;
    $types .= 's';
}

$min_price = filter_input(INPUT_GET, 'min_price', FILTER_VALIDATE_INT);
if ($min_price) {
    $where_clauses[] = "price >= ?";
    $params[] = $min_price;
    $types .= 'i';
}

$max_price = filter_input(INPUT_GET, 'max_price', FILTER_VALIDATE_INT);
if ($max_price) {
    $where_clauses[] = "price <= ?";
    $params[] = $max_price;
    $types .= 'i';
}

$where_sql = "WHERE " . implode(' AND ', $where_clauses);

// 2. Fetch tours based on filters
$stmt = $conn->prepare("SELECT id, title, location, price, image_url FROM tours $where_sql ORDER BY is_featured DESC, title ASC");
if (count($params) > 0) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$tours = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// 3. Fetch unique locations for the filter dropdown
$locations_result = $conn->query("SELECT DISTINCT location FROM tours WHERE status = 'published' AND category = 'Tour' ORDER BY location ASC");
$locations = $locations_result->fetch_all(MYSQLI_ASSOC);
?>

<div class="container my-5">
    <div class="row">

        <div class="col-lg-3">
            <div class="card shadow-sm border-0 sticky-top" style="top: 100px;">
                <div class="card-header bg-light border-0">
                    <h5 class="mb-0"><i class="bi bi-filter me-2"></i>Filter Tours</h5>
                </div>
                <div class="card-body">
                    <form method="GET">
                        <div class="mb-3">
                            <label for="q" class="form-label fw-bold">Search</label>
                            <input type="text" class="form-control" id="q" name="q" placeholder="Tour name..." value="<?php echo htmlspecialchars($search_query); ?>">
                        </div>
                        <div class="mb-3">
                            <label for="location" class="form-label fw-bold">Location</label>
                            <select class="form-select" id="location" name="location">
                                <option value="">All Locations</option>
                                <?php foreach ($locations as $loc): ?>
                                    <option value="<?php echo htmlspecialchars($loc['location']); ?>" <?php if ($location_filter == $loc['location']) echo 'selected'; ?>>
                                        <?php echo htmlspecialchars($loc['location']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label fw-bold">Price Range</label>
                            <div class="d-flex align-items-center">
                                <input type="number" class="form-control" name="min_price" placeholder="Min" value="<?php echo htmlspecialchars($min_price); ?>">
                                <span class="mx-2">-</span>
                                <input type="number" class="form-control" name="max_price" placeholder="Max" value="<?php echo htmlspecialchars($max_price); ?>">
                            </div>
                        </div>
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">Apply Filters</button>
                            <a href="tours.php" class="btn btn-outline-secondary">Clear Filters</a>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <div class="col-lg-9">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2 class="mb-0">All Tours</h2>
                <span class="text-muted"><?php echo count($tours); ?> results found</span>
            </div>
            <div class="row">
                <?php if (empty($tours)): ?>
                    <div class="col-12">
                        <div class="card card-body text-center">
                            <p class="mb-0">No tours found matching your criteria. Try adjusting your filters.</p>
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($tours as $tour): ?>
                    <div class="col-md-6 col-lg-4 mb-4">
                        <div class="card feature-card">
                            <img src="<?php echo htmlspecialchars($tour['image_url']); ?>" class="card-img-top" alt="<?php echo htmlspecialchars($tour['title']); ?>">
                            <div class="card-body d-flex flex-column">
                                <h5 class="card-title"><?php echo htmlspecialchars($tour['title']); ?></h5>
                                <p class="card-text small text-muted flex-grow-1"><i class="bi bi-geo-alt-fill me-1"></i><?php echo htmlspecialchars($tour['location']); ?></p>
                                <div class="d-flex justify-content-between align-items-center">
                                    <span class="fw-bold text-primary fs-5"><?php echo format_price($tour['price']); ?></span>
                                    <a href="tour-details.php?id=<?php echo $tour['id']; ?>" class="btn btn-sm btn-outline-primary">View Details</a>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php 
require_once 'includes/footer.php'; 
?>
