<?php
require_once 'includes/init.php';

if (isset($_SESSION['user_id'])) {
    header("Location: dashboard/");
    exit();
}

$errors = [];
$full_name = '';
$email = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $password_confirm = $_POST['password_confirm'] ?? '';
    $terms = isset($_POST['terms']);

    if (empty($full_name) || empty($email) || empty($password)) {
        $errors[] = "All fields are required.";
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Please enter a valid email address.";
    }
    if (strlen($password) < 8) {
        $errors[] = "Password must be at least 8 characters long.";
    }
    if ($password !== $password_confirm) {
        $errors[] = "Passwords do not match.";
    }
    if (!$terms) {
        $errors[] = "You must agree to the terms of service.";
    }

    if (empty($errors)) {
        $stmt_check = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt_check->bind_param("s", $email);
        $stmt_check->execute();
        if ($stmt_check->get_result()->num_rows > 0) {
            $errors[] = "An account with this email address already exists.";
        }
        $stmt_check->close();
    }

    if (empty($errors)) {
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        $stmt_insert = $conn->prepare("INSERT INTO users (full_name, email, password, role) VALUES (?, ?, ?, 'customer')");
        $stmt_insert->bind_param("sss", $full_name, $email, $password_hash);
        
        if ($stmt_insert->execute()) {
            // --- Send Welcome Email ---
            $subject = "Welcome to " . ($site_settings['site_name'] ?? 'our website') . "!";
            $body = "<h1>Welcome, " . htmlspecialchars($full_name) . "!</h1><p>Thank you for registering. You can now log in to your account and start booking your next adventure.</p><p>Best regards,<br>The Team</p>";
            send_email($email, $subject, $body, $full_name);

            $_SESSION['success_message'] = "Registration successful! You can now log in.";
            header("Location: login.php");
            exit();
        } else {
            $errors[] = "There was an error creating your account. Please try again.";
        }
        $stmt_insert->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Your Account</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <div class="container my-5"><div class="row justify-content-center"><div class="col-lg-7"><div class="text-center mb-4"><h1 class="h2">Create an Account</h1><p class="text-muted">Join us to start planning your next adventure.</p></div><div class="card shadow-sm border-0"><div class="card-body p-4 p-md-5"><?php if (!empty($errors)): ?><div class="alert alert-danger"><?php foreach ($errors as $error): ?><p class="mb-0"><?php echo $error; ?></p><?php endforeach; ?></div><?php endif; ?><form method="POST"><div class="mb-3"><label for="full_name" class="form-label">Full Name</label><input type="text" class="form-control" id="full_name" name="full_name" value="<?php echo htmlspecialchars($full_name); ?>" required></div><div class="mb-3"><label for="email" class="form-label">Email address</label><input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($email); ?>" required></div><div class="row"><div class="col-md-6 mb-3"><label for="password" class="form-label">Password</label><input type="password" class="form-control" id="password" name="password" required></div><div class="col-md-6 mb-3"><label for="password_confirm" class="form-label">Confirm Password</label><input type="password" class="form-control" id="password_confirm" name="password_confirm" required></div></div><div class="form-check mb-4"><input class="form-check-input" type="checkbox" name="terms" id="terms" required><label class="form-check-label" for="terms">I agree to the <a href="terms.php">Terms of Service</a></label></div><div class="d-grid"><button type="submit" class="btn btn-primary btn-lg">Create Account</button></div></form></div></div><div class="text-center mt-4"><p class="text-muted">Already have an account? <a href="login.php">Sign In</a></p></div></div></div></div>
</body>
</html>
