<?php
// process_registration.php
require_once 'includes/db.php';

// Ensure the request method is POST
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    header("Location: register.php");
    exit();
}

// --- SHARED DATA & VALIDATION ---
$registration_type = $_POST['registration_type'] ?? 'customer';
$full_name = trim($_POST['full_name']);
$email = trim($_POST['email']);
$password_input = trim($_POST['password']);
$errors = [];

if (empty($full_name)) { $errors[] = "Full name is required."; }
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) { $errors[] = "A valid email is required."; }
if (strlen($password_input) < 8) { $errors[] = "Password must be at least 8 characters."; }

// --- PARTNER-SPECIFIC DATA & VALIDATION ---
if ($registration_type === 'partner') {
    $business_name = trim($_POST['business_name']);
    if (empty($business_name)) { $errors[] = "Business name is required for partners."; }
}

// Check if email already exists
$stmt_check = $conn->prepare("SELECT id FROM users WHERE email = ?");
$stmt_check->bind_param("s", $email);
$stmt_check->execute();
if ($stmt_check->get_result()->num_rows > 0) {
    $errors[] = "This email address is already registered.";
}
$stmt_check->close();

// If there are any errors, redirect back with the first error message
if (!empty($errors)) {
    $error_query = http_build_query(['error' => $errors[0]]);
    $redirect_page = ($registration_type === 'partner') ? 'partner-register.php' : 'register.php';
    header("Location: {$redirect_page}?{$error_query}");
    exit();
}

// --- DATABASE INSERTION ---
$password_hash = password_hash($password_input, PASSWORD_DEFAULT);
$redirect_page = 'login.php'; // Default redirect
$success_message = 'Registration successful! Please log in to continue.';

if ($registration_type === 'partner') {
    $role = 'tour_partner'; // Or determine from a form select if you have multiple partner types
    $status = 'pending'; // Partners require admin approval
    $stmt = $conn->prepare("INSERT INTO users (full_name, business_name, email, password, role, status) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("ssssss", $full_name, $business_name, $email, $password_hash, $role, $status);
    $success_message = 'Thank you! Your partner application has been submitted for review.';
} else { // Customer registration
    $role = 'customer';
    $status = 'active'; // Customers are active immediately
    $stmt = $conn->prepare("INSERT INTO users (full_name, email, password, role, status) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param("sssss", $full_name, $email, $password_hash, $role, $status);
}

if ($stmt->execute()) {
    // On success, redirect to the login page with a success message
    $success_query = http_build_query(['success' => $success_message]);
    header("Location: {$redirect_page}?{$success_query}");
} else {
    // On failure, redirect back to the form with a generic error
    $error_query = http_build_query(['error' => 'A database error occurred. Please try again.']);
    $redirect_page = ($registration_type === 'partner') ? 'partner-register.php' : 'register.php';
    header("Location: {$redirect_page}?{$error_query}");
}
$stmt->close();
$conn->close();
exit();
?>
