<?php
require_once 'header.php'; 
$page_title = 'My Flight Bookings';

// --- PAGE-SPECIFIC QUERY ---
// Fetches all flight bookings for the logged-in user, joining with the airports table.
$stmt = $conn->prepare("
    SELECT 
        b.id,
        b.booking_ref,
        b.trip_type,
        b.departure_date,
        b.return_date,
        b.num_adults,
        b.num_children,
        b.num_infants,
        b.status,
        orig.name AS origin_name,      -- CORRECTED: Was 'orig.city'
        dest.name AS destination_name  -- CORRECTED: Was 'dest.city'
    FROM 
        flight_bookings b
    JOIN 
        airports orig ON b.origin_id = orig.id
    JOIN 
        airports dest ON b.destination_id = dest.id
    WHERE 
        b.user_id = ?
    ORDER BY 
        b.departure_date DESC
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$bookings = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Helper to get a Bootstrap class based on the booking status
function get_status_badge($status) {
    switch (strtolower($status)) {
        case 'confirmed': return 'bg-success';
        case 'pending': return 'bg-warning text-dark';
        case 'quoted': return 'bg-info';
        case 'cancelled': return 'bg-danger';
        default: return 'bg-secondary';
    }
}
?>

<div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
    <h1 class="h2"><i class="bi bi-airplane-fill me-2"></i>My Flight Bookings</h1>
    <a href="../flights.php" class="btn btn-primary"><i class="bi bi-plus-circle-fill me-1"></i> Book a New Flight</a>
</div>

<div class="card shadow-sm border-0">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-hover align-middle">
                <thead class="table-light">
                    <tr>
                        <th>Reference</th>
                        <th>Route</th>
                        <th>Dates</th>
                        <th class="text-center">Passengers</th>
                        <th class="text-center">Status</th>
                        <th class="text-end">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($bookings)): ?>
                        <tr>
                            <td colspan="6" class="text-center p-5">
                                <i class="bi bi-search fs-2 text-muted"></i>
                                <p class="mt-2 mb-0 text-muted">You have no flight bookings yet.</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($bookings as $booking): ?>
                        <tr>
                            <td class="fw-bold"><?php echo htmlspecialchars($booking['booking_ref'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($booking['origin_name']); ?> &rarr; <?php echo htmlspecialchars($booking['destination_name']); ?></td>
                            <td>
                                <?php echo date('M j, Y', strtotime($booking['departure_date'])); ?>
                                <?php if ($booking['trip_type'] == 'Round-trip' && !empty($booking['return_date'])): ?>
                                    - <?php echo date('M j, Y', strtotime($booking['return_date'])); ?>
                                <?php endif; ?>
                            </td>
                            <td class="text-center"><?php echo $booking['num_adults'] + $booking['num_children'] + $booking['num_infants']; ?></td>
                            <td class="text-center">
                                <span class="badge <?php echo get_status_badge($booking['status']); ?>">
                                    <?php echo htmlspecialchars(ucfirst($booking['status'])); ?>
                                </span>
                            </td>
                            <td class="text-end">
                                <a href="view-flight-booking.php?id=<?php echo $booking['id']; ?>" class="btn btn-sm btn-primary">View Details</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php 
require_once 'footer.php'; 
?>
